unit DebugBox;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, ExtCtrls, StdCtrls;

type
  TPositions = (poTopLeft,poBottomLeft,poTopRight,poBottomRight);

  TDebugBox = class(TComponent)
  private
    DebugForm : TForm;
    DebugList : TListBox;
    FPosition : TPositions;
    FVisible  : Boolean;
    FActive   : Boolean;
    FWidth    : Integer;
    FHeight   : Integer;
    FCaption  : String;
    FLogFile  : string;
    procedure SetPosition(A: TPositions);
    procedure SetVisible(A: Boolean);

    procedure SetWidth(A: Integer);
    procedure SetHeight(A: Integer);
    procedure SetCaption(A: String);
  protected
    { Protected declarations }
    procedure Loaded; override;
  public
    constructor Create(AOwner: TComponent); override;
    procedure Add(A: String);
    procedure Clear;
  published
    property Caption: String read FCaption write SetCaption;
    property Position: TPositions read FPosition write SetPosition default poTopRight;
    property Visible: Boolean read FVisible write SetVisible;
    property Active : Boolean read FActive write FActive;
    property Width: Integer read FWidth write SetWidth default 250;
    property Height: Integer read FHeight write SetHeight default 200;
    Property LogFile : string Read FLogFile Write FLogFile;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('Dialogs', [TDebugBox]);
end;

constructor TDebugBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FPosition := poTopRight;
  FVisible := True;
  FActive  := True;
  FCaption := 'Debug Box';
  FWidth := 250;
  FHeight := 200;
  if not (csDesigning in ComponentState) then
    begin
      DebugForm := TForm.Create(Application);
      with DebugForm do
        begin
          FormStyle := fsStayOnTop;
          BorderStyle := bsSizeable;
          BorderIcons := [biSystemMenu];
        end;
      DebugList := TListBox.Create(DebugForm);
      with DebugList do
        begin
          Parent := DebugForm;
          Align := alClient;
          Sorted := False;
          Font.Name := 'Small Fonts';
          Font.Size := 7;
        end;
    end;
end;

procedure TDebugBox.Loaded;
var F:TextFile;
begin
  inherited Loaded;
      if Active then
      begin
          if (length(LogFile)>0) then
          begin
            AssignFile(F, LogFile);
            Rewrite(F);
            WriteLn(F,'DebugBox Logfile: '+DateToStr(date)+' - '+TimeToStr(time));
            CloseFile(F);
          end;
      end;
end;

procedure TDebugBox.SetPosition(A: TPositions);
begin
  FPosition := A;
  if not (csDesigning in ComponentState) then with DebugForm do
    case A of
      poTopLeft     : SetBounds(0,0,Width,Height);
      poBottomLeft  : SetBounds(0,Screen.Height-Height,Width,Height);
      poTopRight    : SetBounds(Screen.Width-Width,0,Width,Height);
      poBottomRight : SetBounds(Screen.Width-Width,Screen.Height-Height,Width,Height);
    end;
end;

procedure TDebugBox.SetVisible(A: Boolean);
begin
  FVisible := A;
  if not (csDesigning in ComponentState) then
    begin
      DebugForm.Hide;
      if A then
        begin
          Width := Self.Width;
          Height := Self.Height;
          SetPosition(FPosition);
          DebugForm.Show;
        end;
    end;
end;

procedure TDebugBox.SetWidth(A: Integer);
begin
  FWidth := A;
  if not (csDesigning in ComponentState) then
    begin
      DebugForm.Width := FWidth;
      SetPosition(FPosition);
    end;
end;

procedure TDebugBox.SetHeight(A: Integer);
begin
  FHeight := A;
  if not (csDesigning in ComponentState) then
    begin
      DebugForm.Height := FHeight;
      SetPosition(FPosition);
    end;
end;

procedure TDebugBox.SetCaption(A: String);
begin
  FCaption := A;
  if not (csDesigning in ComponentState) then
    DebugForm.Caption := FCaption;
end;

procedure TDebugBox.Add(A: String);
var F:TextFile;
begin
  if Active then
  begin
    if (length(LogFile)>0) then
    begin
        AssignFile(F, LogFile);
        Append(F);
        WriteLn(F,A);
        CloseFile(F);
    end;
    DebugList.Items.Add(A);
    {This makes sure the item just added is visible}
    DebugList.ItemIndex := DebugList.Items.Count-1;
  end;
end;

procedure TDebugBox.Clear;
begin
  if Active then
  begin
    {Remove all items from the list box}
    DebugList.Items.Clear;
  end;
end;


end.
